<?php

namespace App\Controllers\Admin;

use Core\Controller;
use App\Models\Invoice;
use App\Services\Email\EmailService;

class InvoiceController extends Controller
{
    private EmailService $emailService;

    public function __construct()
    {
        parent::__construct();
        $this->emailService = new EmailService();
    }

    public function index(): void
    {
        $status = $this->request->get('status', 'all');
        
        $query = $this->database->table('invoices')
            ->select('invoices.*', 'users.first_name', 'users.last_name', 'users.email')
            ->join('users', 'invoices.user_id', '=', 'users.id');
        
        if ($status !== 'all') {
            $query->where('invoices.status', $status);
        }
        
        $invoices = $query->orderBy('invoices.created_at', 'DESC')->get();
        
        $this->view('admin.invoices.index', [
            'invoices' => $invoices,
            'status' => $status,
        ]);
    }

    public function show(int $id): void
    {
        $invoice = $this->database->table('invoices')
            ->select('invoices.*', 'users.first_name', 'users.last_name', 'users.email', 'users.company', 'users.address', 'users.city', 'users.state', 'users.zip', 'users.country')
            ->join('users', 'invoices.user_id', '=', 'users.id')
            ->where('invoices.id', $id)
            ->first();
        
        if (!$invoice) {
            $this->response->setStatusCode(404);
            $this->view('errors.404');
            return;
        }
        
        // Get invoice items
        $items = $this->database->table('invoice_items')
            ->where('invoice_id', $id)
            ->get();
        
        $this->view('admin.invoices.show', [
            'invoice' => $invoice,
            'items' => $items,
        ]);
    }

    public function create(): void
    {
        $data = $this->validate([
            'user_id' => 'required|integer|exists:users,id',
            'due_date' => 'required|date',
            'items' => 'required',
        ]);

        try {
            $this->database->beginTransaction();
            
            // Generate invoice number
            $invoiceNumber = Invoice::generateInvoiceNumber();
            
            // Calculate totals
            $items = json_decode($data['items'], true);
            $subtotal = 0;
            
            foreach ($items as $item) {
                $subtotal += $item['quantity'] * $item['unit_price'];
            }
            
            $taxRate = (float)$this->request->post('tax_rate', 0);
            $tax = $subtotal * ($taxRate / 100);
            $total = $subtotal + $tax;
            
            // Create invoice
            $invoiceId = $this->database->table('invoices')->insert([
                'user_id' => $data['user_id'],
                'invoice_number' => $invoiceNumber,
                'status' => 'unpaid',
                'subtotal' => $subtotal,
                'tax' => $tax,
                'total' => $total,
                'amount' => $total,
                'due_date' => $data['due_date'],
                'notes' => $this->request->post('notes', ''),
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s'),
            ]);
            
            // Add invoice items
            foreach ($items as $item) {
                $this->database->table('invoice_items')->insert([
                    'invoice_id' => $invoiceId,
                    'description' => $item['description'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'amount' => $item['quantity'] * $item['unit_price'],
                    'created_at' => date('Y-m-d H:i:s'),
                ]);
            }
            
            // Get user info
            $user = $this->database->table('users')->where('id', $data['user_id'])->first();
            
            // Send invoice email
            $invoice = $this->database->table('invoices')->where('id', $invoiceId)->first();
            $this->emailService->sendInvoiceEmail(
                $user['email'],
                $user['first_name'] . ' ' . $user['last_name'],
                $invoice
            );
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'invoice_created',
                'description' => "Created invoice #{$invoiceNumber} for {$user['email']}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            $this->database->commit();

            $this->response
                ->withSuccess('Invoice created successfully')
                ->redirect('/admin/invoices/' . $invoiceId);
        } catch (\Exception $e) {
            $this->database->rollback();
            logger('Invoice creation error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to create invoice']])
                ->withInput()
                ->back();
        }
    }

    public function markPaid(int $id): void
    {
        $invoice = $this->database->table('invoices')->where('id', $id)->first();
        
        if (!$invoice) {
            $this->json(['error' => 'Invoice not found'], 404);
            return;
        }

        $data = $this->validate([
            'payment_method' => 'required',
            'transaction_id' => 'required',
        ]);

        try {
            $this->database->beginTransaction();
            
            // Update invoice
            $this->database->table('invoices')
                ->where('id', $id)
                ->update([
                    'status' => 'paid',
                    'paid_date' => date('Y-m-d H:i:s'),
                    'payment_method' => $data['payment_method'],
                    'transaction_id' => $data['transaction_id'],
                    'updated_at' => date('Y-m-d H:i:s'),
                ]);
            
            // Create transaction record
            $this->database->table('transactions')->insert([
                'user_id' => $invoice['user_id'],
                'invoice_id' => $id,
                'transaction_id' => $data['transaction_id'],
                'gateway' => $data['payment_method'],
                'amount' => $invoice['amount'],
                'currency' => 'USD',
                'status' => 'completed',
                'payment_method' => $data['payment_method'],
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'invoice_marked_paid',
                'description' => "Marked invoice #{$invoice['invoice_number']} as paid",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            $this->database->commit();

            $this->response
                ->withSuccess('Invoice marked as paid successfully')
                ->redirect('/admin/invoices/' . $id);
        } catch (\Exception $e) {
            $this->database->rollback();
            logger('Invoice payment error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to mark invoice as paid']])
                ->back();
        }
    }
}