<?php

namespace App\Controllers\Admin;

use Core\Controller;

class ServerController extends Controller
{
    public function index(): void
    {
        $servers = $this->database->table('servers')
            ->orderBy('created_at', 'DESC')
            ->get();
        
        $this->view('admin.servers.index', [
            'servers' => $servers,
        ]);
    }

    public function show(int $id): void
    {
        $server = $this->database->table('servers')
            ->where('id', $id)
            ->first();
        
        if (!$server) {
            $this->response->setStatusCode(404);
            $this->view('errors.404');
            return;
        }
        
        // Get accounts on this server
        $accounts = $this->database->table('hosting_accounts')
            ->select('hosting_accounts.*', 'users.first_name', 'users.last_name', 'users.email')
            ->join('users', 'hosting_accounts.user_id', '=', 'users.id')
            ->where('hosting_accounts.server_id', $id)
            ->get();
        
        $this->view('admin.servers.show', [
            'server' => $server,
            'accounts' => $accounts,
        ]);
    }

    public function store(): void
    {
        $data = $this->validate([
            'name' => 'required|min:2|max:255',
            'hostname' => 'required|max:255',
            'ip_address' => 'required',
            'type' => 'required|in:shared,vps,dedicated',
            'max_accounts' => 'integer',
            'nameserver1' => 'max:255',
            'nameserver2' => 'max:255',
        ]);

        $data['status'] = 'active';
        $data['current_accounts'] = 0;
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['updated_at'] = date('Y-m-d H:i:s');

        try {
            $serverId = $this->database->table('servers')->insert($data);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'server_created',
                'description' => "Created server: {$data['name']}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);

            $this->response
                ->withSuccess('Server created successfully')
                ->redirect('/admin/servers/' . $serverId);
        } catch (\Exception $e) {
            logger('Server creation error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to create server']])
                ->withInput()
                ->back();
        }
    }

    public function update(int $id): void
    {
        $server = $this->database->table('servers')->where('id', $id)->first();
        
        if (!$server) {
            $this->response->setStatusCode(404);
            $this->view('errors.404');
            return;
        }

        $data = $this->validate([
            'name' => 'required|min:2|max:255',
            'hostname' => 'required|max:255',
            'ip_address' => 'required',
            'type' => 'required|in:shared,vps,dedicated',
            'status' => 'required|in:active,inactive,maintenance',
            'max_accounts' => 'integer',
            'nameserver1' => 'max:255',
            'nameserver2' => 'max:255',
        ]);

        $data['updated_at'] = date('Y-m-d H:i:s');

        try {
            $this->database->table('servers')
                ->where('id', $id)
                ->update($data);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'server_updated',
                'description' => "Updated server: {$data['name']}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);

            $this->response
                ->withSuccess('Server updated successfully')
                ->redirect('/admin/servers/' . $id);
        } catch (\Exception $e) {
            logger('Server update error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to update server']])
                ->withInput()
                ->back();
        }
    }
}