<?php

namespace App\Controllers\Client;

use Core\Controller;

class AccountController extends Controller
{
    public function index(): void
    {
        $user = $this->getAuthUser();
        
        $this->view('client.account.index', [
            'user' => $user,
        ]);
    }

    public function update(): void
    {
        $user = $this->getAuthUser();
        
        $data = $this->validate([
            'first_name' => 'required|min:2|max:50',
            'last_name' => 'required|min:2|max:50',
            'phone' => 'required',
            'company' => 'max:100',
            'address' => 'required',
            'city' => 'required',
            'state' => 'required',
            'zip' => 'required',
            'country' => 'required',
        ]);

        $data['updated_at'] = date('Y-m-d H:i:s');

        try {
            $this->database->table('users')
                ->where('id', $user['id'])
                ->update($data);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $user['id'],
                'action' => 'profile_updated',
                'description' => 'Updated profile information',
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);

            $this->response
                ->withSuccess('Profile updated successfully')
                ->redirect('/client/account');
        } catch (\Exception $e) {
            logger('Profile update error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to update profile']])
                ->withInput()
                ->back();
        }
    }

    public function changePassword(): void
    {
        $user = $this->getAuthUser();
        
        $data = $this->validate([
            'current_password' => 'required',
            'new_password' => 'required|min:8|confirmed',
        ]);

        // Verify current password
        if (!verify_password($data['current_password'], $user['password'])) {
            $this->response
                ->withErrors(['current_password' => ['Current password is incorrect']])
                ->back();
            return;
        }

        try {
            $this->database->table('users')
                ->where('id', $user['id'])
                ->update([
                    'password' => hash_password($data['new_password']),
                    'updated_at' => date('Y-m-d H:i:s'),
                ]);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $user['id'],
                'action' => 'password_changed',
                'description' => 'Changed account password',
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);

            $this->response
                ->withSuccess('Password changed successfully')
                ->redirect('/client/account');
        } catch (\Exception $e) {
            logger('Password change error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to change password']])
                ->back();
        }
    }
}