<?php

namespace App\Models;

use Core\Model;

class Invoice extends Model
{
    protected string $table = 'invoices';
    protected string $primaryKey = 'id';
    
    protected array $fillable = [
        'user_id',
        'invoice_number',
        'status',
        'subtotal',
        'tax',
        'total',
        'amount',
        'due_date',
        'paid_date',
        'payment_method',
        'transaction_id',
        'notes',
    ];
    
    protected array $casts = [
        'id' => 'int',
        'user_id' => 'int',
        'subtotal' => 'float',
        'tax' => 'float',
        'total' => 'float',
        'amount' => 'float',
    ];
    
    protected bool $timestamps = true;

    public function user(): ?array
    {
        return $this->db->table('users')
            ->where('id', $this->user_id ?? 0)
            ->first();
    }

    public function items(): array
    {
        return $this->db->table('invoice_items')
            ->where('invoice_id', $this->id ?? 0)
            ->get();
    }

    public function isPaid(): bool
    {
        $invoice = $this->find($this->id ?? 0);
        return $invoice && $invoice['status'] === 'paid';
    }

    public function isUnpaid(): bool
    {
        $invoice = $this->find($this->id ?? 0);
        return $invoice && $invoice['status'] === 'unpaid';
    }

    public function isOverdue(): bool
    {
        $invoice = $this->find($this->id ?? 0);
        if (!$invoice || $invoice['status'] !== 'unpaid') {
            return false;
        }
        
        return strtotime($invoice['due_date']) < time();
    }

    public function markAsPaid(string $paymentMethod, string $transactionId): bool
    {
        if (!$this->id) {
            return false;
        }
        
        return $this->update($this->id, [
            'status' => 'paid',
            'paid_date' => date('Y-m-d H:i:s'),
            'payment_method' => $paymentMethod,
            'transaction_id' => $transactionId,
        ]) > 0;
    }

    public static function generateInvoiceNumber(): string
    {
        $prefix = 'INV';
        $date = date('Ymd');
        $random = str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        
        return $prefix . '-' . $date . '-' . $random;
    }

    public function addItem(string $description, int $quantity, float $unitPrice): int
    {
        if (!$this->id) {
            return 0;
        }
        
        $amount = $quantity * $unitPrice;
        
        return $this->db->table('invoice_items')->insert([
            'invoice_id' => $this->id,
            'description' => $description,
            'quantity' => $quantity,
            'unit_price' => $unitPrice,
            'amount' => $amount,
            'created_at' => date('Y-m-d H:i:s'),
        ]);
    }
}