<?php

namespace App\Models;

use Core\Model;

class Ticket extends Model
{
    protected string $table = 'tickets';
    protected string $primaryKey = 'id';
    
    protected array $fillable = [
        'user_id',
        'ticket_number',
        'subject',
        'department',
        'priority',
        'status',
    ];
    
    protected array $casts = [
        'id' => 'int',
        'user_id' => 'int',
    ];
    
    protected bool $timestamps = true;

    public function user(): ?array
    {
        return $this->db->table('users')
            ->where('id', $this->user_id ?? 0)
            ->first();
    }

    public function replies(): array
    {
        return $this->db->table('ticket_replies')
            ->where('ticket_id', $this->id ?? 0)
            ->orderBy('created_at', 'ASC')
            ->get();
    }

    public function isOpen(): bool
    {
        $ticket = $this->find($this->id ?? 0);
        return $ticket && $ticket['status'] === 'open';
    }

    public function isClosed(): bool
    {
        $ticket = $this->find($this->id ?? 0);
        return $ticket && $ticket['status'] === 'closed';
    }

    public function close(): bool
    {
        if (!$this->id) {
            return false;
        }
        
        return $this->update($this->id, [
            'status' => 'closed',
        ]) > 0;
    }

    public function reopen(): bool
    {
        if (!$this->id) {
            return false;
        }
        
        return $this->update($this->id, [
            'status' => 'open',
        ]) > 0;
    }

    public function addReply(int $userId, string $message, bool $isStaff = false): int
    {
        if (!$this->id) {
            return 0;
        }
        
        return $this->db->table('ticket_replies')->insert([
            'ticket_id' => $this->id,
            'user_id' => $userId,
            'message' => $message,
            'is_staff' => $isStaff ? 1 : 0,
            'created_at' => date('Y-m-d H:i:s'),
        ]);
    }

    public static function generateTicketNumber(): string
    {
        $prefix = 'TKT';
        $date = date('Ymd');
        $random = str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        
        return $prefix . '-' . $date . '-' . $random;
    }
}