<?php

namespace App\Services\ResellerClub;

use GuzzleHttp\Client;

class ResellerClubService
{
    private string $authUserId;
    private string $apiKey;
    private bool $testMode;
    private Client $client;
    private string $baseUrl;

    public function __construct()
    {
        $this->authUserId = config('services.resellerclub.auth_userid');
        $this->apiKey = config('services.resellerclub.api_key');
        $this->testMode = config('services.resellerclub.test_mode', false);
        
        $this->baseUrl = $this->testMode 
            ? 'https://test.httpapi.com/api/'
            : 'https://httpapi.com/api/';
        
        $this->client = new Client([
            'base_uri' => $this->baseUrl,
        ]);
    }

    private function buildQuery(array $params): array
    {
        return array_merge([
            'auth-userid' => $this->authUserId,
            'api-key' => $this->apiKey,
        ], $params);
    }

    public function checkAvailability(string $domain): array
    {
        try {
            $parts = explode('.', $domain);
            $sld = $parts[0];
            $tld = implode('.', array_slice($parts, 1));
            
            $response = $this->client->get('domains/available.json', [
                'query' => $this->buildQuery([
                    'domain-name' => $sld,
                    'tlds' => $tld,
                ]),
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            return [
                'success' => true,
                'available' => $result[$domain]['status'] === 'available',
            ];
        } catch (\Exception $e) {
            logger('ResellerClub checkAvailability error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function registerDomain(array $params): array
    {
        try {
            $response = $this->client->post('domains/register.json', [
                'form_params' => $this->buildQuery($params),
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if (isset($result['status']) && $result['status'] === 'Success') {
                return [
                    'success' => true,
                    'order_id' => $result['entityid'],
                    'data' => $result,
                ];
            }
            
            return [
                'success' => false,
                'error' => $result['message'] ?? 'Registration failed',
            ];
        } catch (\Exception $e) {
            logger('ResellerClub registerDomain error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function renewDomain(int $orderId, int $years): array
    {
        try {
            $response = $this->client->post('domains/renew.json', [
                'form_params' => $this->buildQuery([
                    'order-id' => $orderId,
                    'years' => $years,
                    'exp-date' => time(),
                    'invoice-option' => 'NoInvoice',
                ]),
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if (isset($result['status']) && $result['status'] === 'Success') {
                return [
                    'success' => true,
                    'data' => $result,
                ];
            }
            
            return [
                'success' => false,
                'error' => $result['message'] ?? 'Renewal failed',
            ];
        } catch (\Exception $e) {
            logger('ResellerClub renewDomain error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function updateNameservers(int $orderId, array $nameservers): array
    {
        try {
            $params = [
                'order-id' => $orderId,
            ];
            
            foreach ($nameservers as $index => $ns) {
                $params['ns'][] = $ns;
            }
            
            $response = $this->client->post('domains/modify-ns.json', [
                'form_params' => $this->buildQuery($params),
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if (isset($result['status']) && $result['status'] === 'Success') {
                return [
                    'success' => true,
                    'message' => 'Nameservers updated successfully',
                ];
            }
            
            return [
                'success' => false,
                'error' => $result['message'] ?? 'Update failed',
            ];
        } catch (\Exception $e) {
            logger('ResellerClub updateNameservers error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getDomainDetails(int $orderId): array
    {
        try {
            $response = $this->client->get('domains/details.json', [
                'query' => $this->buildQuery([
                    'order-id' => $orderId,
                    'options' => 'All',
                ]),
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            return [
                'success' => true,
                'data' => $result,
            ];
        } catch (\Exception $e) {
            logger('ResellerClub getDomainDetails error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function transferDomain(array $params): array
    {
        try {
            $response = $this->client->post('domains/transfer.json', [
                'form_params' => $this->buildQuery($params),
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if (isset($result['status']) && $result['status'] === 'Success') {
                return [
                    'success' => true,
                    'order_id' => $result['entityid'],
                    'data' => $result,
                ];
            }
            
            return [
                'success' => false,
                'error' => $result['message'] ?? 'Transfer failed',
            ];
        } catch (\Exception $e) {
            logger('ResellerClub transferDomain error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }
}