<?php

namespace App\Services\WHM;

use GuzzleHttp\Client;

class WHMService
{
    private string $host;
    private string $username;
    private string $apiToken;
    private int $port;
    private Client $client;

    public function __construct()
    {
        $this->host = config('services.whm.host');
        $this->username = config('services.whm.username');
        $this->apiToken = config('services.whm.api_token');
        $this->port = config('services.whm.port', 2087);
        
        $this->client = new Client([
            'base_uri' => "https://{$this->host}:{$this->port}/json-api/",
            'verify' => false, // In production, use proper SSL verification
            'headers' => [
                'Authorization' => "whm {$this->username}:{$this->apiToken}",
            ],
        ]);
    }

    public function createAccount(array $params): array
    {
        try {
            $response = $this->client->post('createacct', [
                'form_params' => $params,
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($result['metadata']['result'] == 1) {
                return [
                    'success' => true,
                    'data' => $result['data'],
                ];
            }
            
            return [
                'success' => false,
                'error' => $result['metadata']['reason'] ?? 'Unknown error',
            ];
        } catch (\Exception $e) {
            logger('WHM createAccount error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function suspendAccount(string $username, string $reason = ''): array
    {
        try {
            $response = $this->client->post('suspendacct', [
                'form_params' => [
                    'user' => $username,
                    'reason' => $reason,
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($result['metadata']['result'] == 1) {
                return [
                    'success' => true,
                    'message' => 'Account suspended successfully',
                ];
            }
            
            return [
                'success' => false,
                'error' => $result['metadata']['reason'] ?? 'Unknown error',
            ];
        } catch (\Exception $e) {
            logger('WHM suspendAccount error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function unsuspendAccount(string $username): array
    {
        try {
            $response = $this->client->post('unsuspendacct', [
                'form_params' => [
                    'user' => $username,
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($result['metadata']['result'] == 1) {
                return [
                    'success' => true,
                    'message' => 'Account unsuspended successfully',
                ];
            }
            
            return [
                'success' => false,
                'error' => $result['metadata']['reason'] ?? 'Unknown error',
            ];
        } catch (\Exception $e) {
            logger('WHM unsuspendAccount error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function terminateAccount(string $username, bool $keepDns = false): array
    {
        try {
            $response = $this->client->post('removeacct', [
                'form_params' => [
                    'user' => $username,
                    'keepdns' => $keepDns ? 1 : 0,
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($result['metadata']['result'] == 1) {
                return [
                    'success' => true,
                    'message' => 'Account terminated successfully',
                ];
            }
            
            return [
                'success' => false,
                'error' => $result['metadata']['reason'] ?? 'Unknown error',
            ];
        } catch (\Exception $e) {
            logger('WHM terminateAccount error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function changePassword(string $username, string $password): array
    {
        try {
            $response = $this->client->post('passwd', [
                'form_params' => [
                    'user' => $username,
                    'password' => $password,
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($result['metadata']['result'] == 1) {
                return [
                    'success' => true,
                    'message' => 'Password changed successfully',
                ];
            }
            
            return [
                'success' => false,
                'error' => $result['metadata']['reason'] ?? 'Unknown error',
            ];
        } catch (\Exception $e) {
            logger('WHM changePassword error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getAccountInfo(string $username): array
    {
        try {
            $response = $this->client->get('accountsummary', [
                'query' => [
                    'user' => $username,
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($result['metadata']['result'] == 1) {
                return [
                    'success' => true,
                    'data' => $result['data']['acct'][0] ?? [],
                ];
            }
            
            return [
                'success' => false,
                'error' => $result['metadata']['reason'] ?? 'Unknown error',
            ];
        } catch (\Exception $e) {
            logger('WHM getAccountInfo error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }
}